<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\Attendance;
use App\Models\Leave;
use App\Models\Holiday;
use Carbon\Carbon;

class DeductDailyAbsentLeaves extends Command
{
    protected $signature = 'leaves:deduct-absent';
    protected $description = 'Deduct annual leave for absent users not on leave, weekend, or holiday';

    public function handle()
    {
        $today = Carbon::today()->toDateString();
        \Log::info("Cron is working fine!");

        // Skip weekends
        if (Carbon::parse($today)->isWeekend()) {
            $this->info("Today is a weekend. Skipping.");
            return;
        }

        // Skip DB-based holidays
        if (Holiday::whereDate('date', $today)->exists()) {
            $this->info("Today is a holiday. Skipping.");
            return;
        }

        $users = User::where('role','employee')->get();

        foreach ($users as $user) {
            // Skip if attendance exists
            if (Attendance::where('user_id', $user->id)->whereDate('date', $today)->exists()) {
                continue;
            }

            // Skip if on approved leave
            $onLeave = Leave::where('user_id', $user->id)
                ->where('status', 'approved')
                ->where('start_date', '<=', $today)
                ->where('end_date', '>=', $today)
                ->exists();

            if ($onLeave) {
                continue;
            }

            // Check how much annual leave used
            $usedAnnual = Leave::where('user_id', $user->id)
                ->where('type', 'annual')
                ->where('status', 'approved')
                ->whereYear('start_date', now()->year)
                ->sum('days');

            if ($usedAnnual < 14) {
                Leave::create([
                    'user_id' => $user->id,
                    'type' => 'annual',
                    'start_date' => $today,
                    'end_date' => $today,
                    'days' => 1,
                    'status' => 'approved',
                    'reason' => 'Auto deducted for being absent',
                ]);
                

                $this->info("Deducted 1 annual leave from {$user->name}");
            } else {
                 Leave::create([
                    'user_id' => $user->id,
                    'type' => 'nopay',
                    'start_date' => $today,
                    'end_date' => $today,
                    'days' => 1,
                    'status' => 'approved',
                    'reason' => 'nopay',
                ]);
                $this->warn("No leave left for {$user->name}");
            }
        }

        $this->info("Daily absence check completed.");
    }
}